#include "StdAfx.h"
#include "LazyWindow.hpp"
#include <time.h>

namespace DSLib
{

	HINSTANCE LazyWindow::hInstance = NULL;

	static ATOM sWindowClass = 0;
	static ATOM sWindowDataAtom = 0;
	static LONG volatile sInitializeCount = 0;

	bool LazyWindow::Initialize()
	{
		if(InterlockedIncrement(&sInitializeCount) != 1)
			return true;

		if(sWindowClass) return true;
		if(hInstance == NULL) hInstance = GetModuleHandle(NULL);

		WNDCLASSEX wndClassEx;

		wndClassEx.cbSize = sizeof(WNDCLASSEX);
		wndClassEx.style = CS_DBLCLKS | CS_OWNDC;
		wndClassEx.lpfnWndProc = LazyWindow::WindowProc;
		wndClassEx.cbClsExtra = 0;
		wndClassEx.cbWndExtra = 0;
		wndClassEx.hInstance = hInstance;
		wndClassEx.hIcon = NULL;
		wndClassEx.hCursor = LoadCursor(NULL, IDC_ARROW);
		wndClassEx.hbrBackground = (HBRUSH)COLOR_WINDOW;
		wndClassEx.lpszMenuName = NULL;

		std::tstringstream ss;
		ss << _T("ZzLab.WndClass.") << (int)clock();
		std::tstring name = ss.str();
		wndClassEx.lpszClassName = name.c_str();

		wndClassEx.hIconSm = NULL;

		sWindowClass = RegisterClassEx(&wndClassEx);
		if(! sWindowClass)
		{
			DWORD err = GetLastError();
			_ERROR_FUNC0(_T("err=") << err);
		}

		sWindowDataAtom = GlobalAddAtom(_T("ZzLab.WindowData"));
		if(! sWindowDataAtom)
		{
			DWORD err = GetLastError();
			_ERROR_FUNC0(_T("err=") << err);
		}

		return sWindowClass != 0;
	}

	void LazyWindow::Uninitialize()
	{
		if(InterlockedDecrement(&sInitializeCount) != 0)
			return;

		UnregisterClass(MAKEINTATOM(sWindowClass), hInstance);
		sWindowClass = 0;

		GlobalDeleteAtom(sWindowDataAtom);
		sWindowDataAtom = 0;
	}

	void LazyWindow::MessageLoop(MSG& msg)
	{
		BOOL bRet;

		while((bRet = GetMessage(&msg, NULL, 0, 0)) != 0)
		{
			if(bRet == -1)
			{
				DWORD err = GetLastError();
				_ERROR(_T("GetMesasge failed, err=") << err << std::endl);
			}
			else
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}
	}

	bool LazyWindow::MessagePump(MSG& msg)
	{
		if(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
		{
			TranslateMessage(&msg);
			DispatchMessage(&msg);

			if(msg.message == WM_QUIT)
				return false;
		}
		else
			msg.message = WM_NULL;

		return true;
	}

	bool LazyWindow::MessageLoop(MSG& msg, DWORD nCount, const HANDLE* pHandles, DWORD dwMilliseconds, DWORD dwWaitMask, DWORD& res)
	{
		res = MsgWaitForMultipleObjects(nCount, pHandles, FALSE, dwMilliseconds, dwWaitMask);
		if(res == WAIT_OBJECT_0 + nCount)
		{
			while(PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
			{
				TranslateMessage(&msg);
				DispatchMessage(&msg);

				if(msg.message == WM_QUIT)
					return false;
			}				
		}

		return true;
	}

	struct _CREATE_DATA {
		LazyWindow::MSGPROC msgProc;
		intptr_t userData;

		_CREATE_DATA()
		{
			_TRACE_THIS();
		}

		~_CREATE_DATA()
		{
			_TRACE_THIS();
		}
	};

	LRESULT CALLBACK DefMsgProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam, intptr_t)
	{
		return DefWindowProc(hwnd, uMsg, wParam, lParam);
	}

	HWND LazyWindow::CreateMessageOnlyWindow(MSGPROC msgProc, intptr_t userData)
	{
		_CREATE_DATA* data = new _CREATE_DATA;
		data->msgProc = msgProc;
		data->userData = userData;

		HWND hWnd = CreateWindowEx(0, (LPCTSTR)sWindowClass, NULL, 0, 0, 0, 0, 0, HWND_MESSAGE, NULL, hInstance, data);
		if(! hWnd) 
		{
			DWORD err = GetLastError();
			_ERROR_FUNC0(_T("err=") << err);
			return NULL;
		}

		return hWnd;
	}

	HWND LazyWindow::CreateUserWindow(int x, int y, int w, int h, HWND parent, const std::tstring& title, DWORD dwStyle, DWORD dwExStyle, MSGPROC msgProc, intptr_t userData)
	{
		if(msgProc == NULL) msgProc = DefMsgProc;

		_CREATE_DATA* data = new _CREATE_DATA;
		data->msgProc = msgProc;
		data->userData = userData;

		HWND hWnd = CreateWindowEx(dwExStyle, (LPCTSTR)sWindowClass, title.c_str(), dwStyle, x, y, w, h, parent, NULL, hInstance, data);
		if(! hWnd) 
		{
			DWORD err = GetLastError();
			_ERROR_FUNC0(_T("err=") << err);
			return NULL;
		}

		return hWnd;
	}

	LRESULT CALLBACK LazyWindow::WindowProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
	{
		switch(uMsg)
		{
		case WM_NCCREATE:
			{
				CREATESTRUCT* cs = (CREATESTRUCT*)lParam;

				_CREATE_DATA* data = (_CREATE_DATA*)cs->lpCreateParams;
				SetProp(hwnd, MAKEINTATOM(sWindowDataAtom), data);

				return data->msgProc(hwnd, uMsg, wParam, lParam, data->userData);
			}
			break;

		case WM_NCDESTROY:
			{	
				_CREATE_DATA* data = (_CREATE_DATA*)GetProp(hwnd, MAKEINTATOM(sWindowDataAtom));
				if(data)
				{
					LRESULT res = data->msgProc(hwnd, uMsg, wParam, lParam, data->userData);
					delete data;

					return res;
				}
			}
			break;

		default:
			{	
				_CREATE_DATA* data = (_CREATE_DATA*)GetProp(hwnd, MAKEINTATOM(sWindowDataAtom));
				if(data)
					return data->msgProc(hwnd, uMsg, wParam, lParam, data->userData);
			}
			break;
		}

		return DefWindowProc(hwnd, uMsg, wParam, lParam);
	}

	Window::Window() : x(CW_USEDEFAULT), y(CW_USEDEFAULT), width(CW_USEDEFAULT), height(CW_USEDEFAULT), 
		parent(NULL), style(WS_CHILD), exStyle(0), mWnd(NULL)
	{
		_TRACE_THIS();
	}

	Window::~Window()
	{
		_TRACE_THIS();

		Destroy();
	}

	void Window::Create()
	{
		if(mWnd) return;

		LazyWindow::CreateUserWindow(x, y, width, height, parent, title, style, exStyle, _Default_MsgProc, (intptr_t)this);
	}

	void Window::Destroy()
	{
		if(! mWnd) return;

		DestroyWindow(mWnd);
		mWnd = NULL;
	}

	bool Window::OnMessage(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam, LRESULT& lResult)
	{
		switch(uMsg)
		{
		case WM_CREATE:
			mWnd = hwnd;
			break;

		case WM_DESTROY:
			mWnd = NULL;
			break;
		}

		return false;
	}

	LRESULT Window::Default_MsgProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
	{
		LRESULT lResult = 0;
		if(OnMessage(hwnd, uMsg, wParam, lParam, lResult))
			return lResult;

		return DefWindowProc(hwnd, uMsg, wParam, lParam);
	}

	CommandRequest::CommandRequest(HANDLE evt) : completeNotify(evt)
	{
		//_TRACE_THIS();
	}

	CommandRequest::~CommandRequest()
	{
		//_TRACE_THIS();
	}

	enum
	{
		WM_DSLIB_COMMAND = WM_USER + 0x1000,
	};

	Service::Service(void) : mMessageWindow(NULL), mThread(NULL), mThreadReady(NULL)
	{
		_TRACE_THIS();

		mThreadReady = CreateEvent(NULL, FALSE, FALSE, NULL);
	}

	Service::~Service(void)
	{
		_TRACE_THIS();

		Destroy();

		if(mThreadReady)
		{
			CloseHandle(mThreadReady);
			mThreadReady = NULL;
		}
	}

	void Service::Create()
	{
		if(mThread) return;

		mThread = CreateThread(NULL, 0, _ThreadProc, this, 0, &mThreadId);
		WaitForSingleObject(mThreadReady, INFINITE);
	}

	void Service::Destroy()
	{
		if(! mThread) return;

		PostMessageGuaranteed(mMessageWindow, WM_CLOSE, 0, 0);
		WaitForSingleObject(mThread, INFINITE);
		CloseHandle(mThread);
		mThread = NULL;
	}

	void Service::PostCommand(UINT cmd, CommandRequest* req)
	{
		PostMessageGuaranteed(mMessageWindow, WM_DSLIB_COMMAND + cmd, (WPARAM)req, 0);
	}

	void Service::SendCommand(UINT cmd, CommandRequest* req)
	{
		HANDLE tmp = NULL;
		if(req == NULL)
		{
			tmp = CreateEvent(NULL, FALSE, FALSE, NULL);
			req = new CommandRequest(tmp);
		}

		PostMessageGuaranteed(mMessageWindow, WM_DSLIB_COMMAND + cmd, (WPARAM)req, 0);
		WaitForSingleObject(req->completeNotify, INFINITE);

		if(tmp) CloseHandle(tmp); 
	}

	void Service::PostMessageGuaranteed(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
	{
		while(! ::PostMessage(hwnd, uMsg, wParam, lParam))
		{
			DWORD err = GetLastError();
			_ERROR(_T("PostMessageGuaranteed FAILED, err=") << err << std::endl);

			Sleep(100);
		}
	}

	void Service::OnStart()
	{
		_TRACE_THIS();
	}

	void Service::OnStop()
	{
		_TRACE_THIS();
	}

	void Service::OnCreate()
	{
		_TRACE_THIS();
	}

	void Service::OnDestroy()
	{
		_TRACE_THIS();
	}

	void Service::MessageLoop()
	{
		LazyWindow::MessageLoop(mCurrentMessage);
	}

	DWORD WINAPI Service::_ThreadProc(LPVOID lpParameter)
	{
		Service* pThis = (Service*)lpParameter;

		return pThis->ThreadProc();
	}

	DWORD Service::ThreadProc()
	{
		_TRACE_THIS0(_T("LazyWindow service begin."));

		OnStart();
		mMessageWindow = LazyWindow::CreateMessageOnlyWindow(_MsgProc, (intptr_t)this);
		SetEvent(mThreadReady);

		MessageLoop();
		OnStop();

		_TRACE_THIS0(_T("LazyWindow service ended."));
		return (DWORD)mCurrentMessage.wParam;
	}

	LRESULT CALLBACK Service::_MsgProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam, intptr_t userData)
	{
		Service* pThis = (Service*)userData;

		return pThis->MsgProc(hwnd, uMsg, wParam, lParam);
	}

	LRESULT Service::MsgProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
	{
		if(uMsg >= WM_DSLIB_COMMAND)
		{
			CommandRequest* req = (CommandRequest*)wParam;
			LRESULT res = OnCommand(uMsg - WM_DSLIB_COMMAND, req);
			if(req) 
			{
				if(req->completeNotify) SetEvent(req->completeNotify);
				delete req;
			}

			return res;
		}

		switch(uMsg)
		{
		case WM_CREATE:
			OnCreate();
			break;

		case WM_CLOSE:
			{
				MSG msg;
				while(PeekMessage(&msg, hwnd, WM_DSLIB_COMMAND, WM_DSLIB_COMMAND, PM_REMOVE))
				{
					TranslateMessage(&msg);
					DispatchMessage(&msg);
				}

				OnDestroy();

				DestroyWindow(mMessageWindow);
				mMessageWindow = NULL;
				PostQuitMessage(0);
				return 0;
			}
			break;
		}

		return DefWindowProc(hwnd, uMsg, wParam, lParam);
	}

}